r"""
Colorbar
========

The :meth:`pygmt.Figure.colorbar` method creates a color scalebar. The colormap is set
via the ``cmap`` parameter. A full list of available color palette tables can be found
at :gmt-docs:`reference/cpts.html`. Use the ``frame`` parameter to add labels to the
**x** and **y** axes of the colorbar by appending **+l** followed by the desired text.
To add and adjust the annotations (**a**) and ticks (**f**) append the letter followed
by the desired interval. The placement of the colorbar is set by passing a
:class:`pygmt.params.Position` object to the ``position`` parameter.
"""

# %%
import pygmt
from pygmt.params import Position

fig = pygmt.Figure()
fig.basemap(region=[0, 3, 6, 9], projection="x3c", frame=["af", "WSne+tColorbars"])

# ============
# Create a colorbar designed for seismic tomography - roma
# Colorbar is placed at Bottom Center (BC) by default if no position is given
# Add quantity and unit as labels ("+l") to the x and y axes
# Add annotations ("+a") in steps of 0.5 and ticks ("+f") in steps of 0.1
fig.colorbar(cmap="SCM/roma", frame=["xa0.5f0.1+lVelocity", "y+lm/s"])

# ============
# Create a colorbar showing the scientific rainbow - batlow
fig.colorbar(
    cmap="SCM/batlow",
    # A horizontal colorbar positioned at map coordinates (0.3, 8.7), with a
    # length of 4 cm and a width of 0.5 cm.
    position=Position((0.3, 8.7), cstype="mapcoords"),
    length=4,
    width=0.5,
    orientation="horizontal",
    box=True,
    frame=["x+lTemperature", "y+l°C"],
    scale=100,
)

# ============
# Create a colorbar suitable for surface topography - oleron
fig.colorbar(
    cmap="SCM/oleron",
    # Colorbar placed at Middle Right (MR) outside the plot bounding box, offset by 1 cm
    # horizontally and 0 cm vertically from anchor point, with a length of 7 cm and
    # width of 0.5 cm, and a rectangle for NaN values.
    # Note that the label 'Elevation' is moved to the opposite side and plotted
    # vertically as a column of characters.
    position=Position("MR", cstype="outside", offset=(1, 0)),
    length=7,
    width=0.5,
    nan=True,
    move_text=["annotations", "label"],
    label_as_column=True,
    frame=["x+lElevation", "y+lm"],
    scale=10,
)

# ============
# Create a colorbar suitable for categorical data - hawaii
# Set up the colormap
pygmt.makecpt(
    cmap="SCM/hawaii",
    series=[0, 3, 1],
    # Comma-separated string for the annotations of the colorbar
    color_model="+cA,B,C,D",
)
# Plot the colorbar
fig.colorbar(
    cmap=True,  # Use colormap set up above
    # Colorbar placed in the Bottom Left (BL) corner inside the plot bounding box, with
    # an offset by 0.5 cm horizontally and 0.8 cm vertically from the anchor point, and
    # plotted horizontally.
    position=Position("BL", offset=(0.5, 0.8)),
    orientation="horizontal",
    box=True,
    # Divide colorbar into equal-sized rectangles
    equalsize=0.5,
)

fig.show()
